jQuery(document).ready(function($) {
    
    // Test Connection
    $('#test-connection').on('click', function() {
        const btn = $(this);
        const apiUrl = $('#api_url').val();
        const apiKey = $('#api_key').val();
        const resultDiv = $('#connection-result');
        
        if (!apiUrl || !apiKey) {
            resultDiv.html('<div class="notice notice-error"><p>Please enter both API URL and API Key</p></div>');
            return;
        }
        
        btn.prop('disabled', true).text('Testing...');
        resultDiv.html('<div class="notice notice-info"><p>Connecting to API...</p></div>');
        
        $.ajax({
            url: tinysharpData.ajax_url,
            method: 'POST',
            data: {
                action: 'tinysharp_test_connection',
                nonce: tinysharpData.nonce,
                api_url: apiUrl,
                api_key: apiKey
            },
            success: function(response) {
                if (response.success) {
                    const data = response.data.data;
                    resultDiv.html(`
                        <div class="notice notice-success">
                            <p><strong>Connection Successful!</strong></p>
                            <ul>
                                <li>Plan: <strong>${data.plan.toUpperCase()}</strong></li>
                                <li>Total Tokens: <strong>${data.total_tokens.toLocaleString()}</strong></li>
                                <li>Used Tokens: <strong>${data.used_tokens.toLocaleString()}</strong></li>
                                <li>Remaining: <strong>${data.remaining_tokens.toLocaleString()}</strong></li>
                            </ul>
                        </div>
                    `);
                } else {
                    resultDiv.html(`<div class="notice notice-error"><p><strong>Connection Failed:</strong> ${response.data.message}</p></div>`);
                }
            },
            error: function() {
                resultDiv.html('<div class="notice notice-error"><p><strong>Error:</strong> Could not connect to server</p></div>');
            },
            complete: function() {
                btn.prop('disabled', false).text('Test Connection');
            }
        });
    });
    
    // Save Settings
    $('#tinysharp-settings-form').on('submit', function(e) {
        e.preventDefault();
        
        const formData = $(this).serialize();
        const submitBtn = $(this).find('button[type="submit"]');
        
        submitBtn.prop('disabled', true).text('Saving...');
        
        $.ajax({
            url: tinysharpData.ajax_url,
            method: 'POST',
            data: formData + '&action=tinysharp_save_settings&nonce=' + tinysharpData.nonce,
            success: function(response) {
                if (response.success) {
                    $('<div class="notice notice-success is-dismissible"><p>' + response.data.message + '</p></div>')
                        .insertAfter('h1')
                        .delay(3000)
                        .fadeOut();
                } else {
                    alert('Error: ' + response.data.message);
                }
            },
            error: function() {
                alert('Error: Could not save settings');
            },
            complete: function() {
                submitBtn.prop('disabled', false).text('Save Settings');
            }
        });
    });
    
    // Bulk Optimization
    let currentIndex = 0;
    let totalOptimized = 0;
    let totalSaved = 0;
    let tokensUsed = 0;
    let tokensRemaining = typeof initialTokensRemaining !== 'undefined' ? initialTokensRemaining : 0;
    let failedImages = [];
    let isPaused = false;
    
    $('#start-bulk-optimize').on('click', function() {
        if (typeof bulkImageIds === 'undefined' || bulkImageIds.length === 0) {
            alert('No images to optimize');
            return;
        }
        
        // Double-check tokens before starting
        if (tokensRemaining <= 0) {
            alert('You have no tokens remaining. Please purchase additional tokens.');
            window.location.reload();
            return;
        }
        
        $(this).hide();
        $('#bulk-progress').show();
        $('#progress-total').text(bulkImageIds.length);
        
        currentIndex = 0;
        totalOptimized = 0;
        totalSaved = 0;
        tokensUsed = 0;
        failedImages = [];
        
        optimizeNext();
    });
    
    function optimizeNext() {
        if (isPaused) {
            return;
        }
        
        if (currentIndex >= bulkImageIds.length) {
            showResults();
            return;
        }
        
        // Check if we still have tokens
        if (tokensRemaining <= 0) {
            isPaused = true;
            showOutOfTokensError();
            return;
        }
        
        const attachmentId = bulkImageIds[currentIndex];
        
        // Update progress
        $('#progress-current').text(currentIndex + 1);
        $('#current-image').text('Processing image #' + attachmentId);
        
        const progressPercent = ((currentIndex + 1) / bulkImageIds.length) * 100;
        $('#progress-bar').css('width', progressPercent + '%');
        
        $.ajax({
            url: tinysharpData.ajax_url,
            method: 'POST',
            data: {
                action: 'tinysharp_bulk_optimize',
                nonce: tinysharpData.nonce,
                attachment_id: attachmentId
            },
            timeout: 90000,
            success: function(response) {
                if (response.success) {
                    totalOptimized++;
                    totalSaved += response.data.saved || 0;
                    tokensUsed++;
                    tokensRemaining--;
                    
                    $('#total-saved').text(formatBytes(totalSaved));
                    $('#tokens-used').text(tokensUsed);
                    $('#tokens-remaining').text(tokensRemaining);
                    
                    console.log('Optimized image #' + attachmentId + ' - Saved: ' + formatBytes(response.data.saved));
                } else {
                    const errorMessage = response.data.message || response.data || 'Unknown error';
                    
                    if (errorMessage.includes('Insufficient tokens') || 
                        errorMessage.includes('insufficient_tokens') ||
                        errorMessage.includes('Payment Required')) {
                        
                        isPaused = true;
                        showOutOfTokensError();
                        return;
                    }
                    
                    failedImages.push({
                        id: attachmentId,
                        error: errorMessage
                    });
                    console.log('Failed to optimize image #' + attachmentId);
                }
                
                currentIndex++;
                setTimeout(optimizeNext, 500);
            },
            error: function(xhr, status, error) {
                if (xhr.status === 402) {
                    isPaused = true;
                    showOutOfTokensError();
                    return;
                }
                
                failedImages.push({
                    id: attachmentId,
                    error: error || 'Connection error'
                });
                console.log('Failed to optimize image #' + attachmentId);
                
                currentIndex++;
                setTimeout(optimizeNext, 500);
            }
        });
    }
    
    function showOutOfTokensError() {
        $('#bulk-progress').hide();
        
        // Properly extract base URL
        const baseUrl = tinysharpData.api_url.replace(/\/wp-json\/sio\/v1.*/, '');
        const pricingUrl = baseUrl + '/pricing-plans/';
        
        $('#bulk-results').html(`
            <div class="bulk-error-card">
                <div class="error-icon">⚠️</div>
                <h2>Insufficient Tokens</h2>
                <p>You have run out of optimization tokens. Please purchase additional tokens to continue.</p>
                
                <div class="error-stats">
                    <div class="stat">
                        <strong>${totalOptimized}</strong>
                        <span>Images Optimized</span>
                    </div>
                    <div class="stat">
                        <strong>${formatBytes(totalSaved)}</strong>
                        <span>Total Saved</span>
                    </div>
                    <div class="stat">
                        <strong>${tokensUsed}</strong>
                        <span>Tokens Used</span>
                    </div>
                </div>
                
                <div class="error-actions">
                    <a href="${pricingUrl}" 
                    class="button button-primary button-hero" 
                    target="_blank">
                        Purchase More Tokens
                    </a>
                    <button onclick="location.reload()" class="button">
                        Refresh Page
                    </button>
                </div>
            </div>
        `).show();
        
        if (!$('#tinysharp-error-styles').length) {
            $('head').append(`
                <style id="tinysharp-error-styles">
                    .bulk-error-card {
                        background: white;
                        border: 2px solid #dc3545;
                        border-radius: 12px;
                        padding: 40px;
                        text-align: center;
                        max-width: 600px;
                        margin: 40px auto;
                    }
                    .bulk-error-card .error-icon {
                        font-size: 4em;
                        margin-bottom: 20px;
                    }
                    .bulk-error-card h2 {
                        color: #dc3545;
                        margin-bottom: 10px;
                    }
                    .bulk-error-card p {
                        font-size: 1.1em;
                        color: #666;
                        margin-bottom: 30px;
                    }
                    .error-stats {
                        display: grid;
                        grid-template-columns: repeat(3, 1fr);
                        gap: 20px;
                        margin: 30px 0;
                        padding: 20px;
                        background: #f8f9fa;
                        border-radius: 8px;
                    }
                    .error-stats .stat {
                        display: flex;
                        flex-direction: column;
                        gap: 5px;
                    }
                    .error-stats .stat strong {
                        font-size: 1.5em;
                        color: #0969da;
                    }
                    .error-stats .stat span {
                        font-size: 0.9em;
                        color: #666;
                    }
                    .error-actions {
                        display: flex;
                        gap: 15px;
                        justify-content: center;
                        margin-top: 30px;
                    }
                </style>
            `);
        }
    }
    
    function showResults() {
        $('#bulk-progress').hide();
        
        const avgSaved = totalOptimized > 0 ? totalSaved / totalOptimized : 0;
        
        let html = `
            <div class="bulk-results-card">
                <div class="results-icon">✅</div>
                <h2>Bulk Optimization Complete!</h2>
                
                <div class="results-stats">
                    <div class="stat">
                        <strong>${totalOptimized}</strong>
                        <span>Images Optimized</span>
                    </div>
                    <div class="stat">
                        <strong>${formatBytes(totalSaved)}</strong>
                        <span>Total Space Saved</span>
                    </div>
                    <div class="stat">
                        <strong>${formatBytes(avgSaved)}</strong>
                        <span>Average per Image</span>
                    </div>
                    <div class="stat">
                        <strong>${tokensUsed}</strong>
                        <span>Tokens Used</span>
                    </div>
                </div>
        `;
        
        if (failedImages.length > 0) {
            html += `
                <div class="failed-images">
                    <h3>Failed Images (${failedImages.length})</h3>
                    <ul>
            `;
            failedImages.forEach(function(img) {
                html += `<li>Image #${img.id}: ${img.error}</li>`;
            });
            html += `</ul></div>`;
        }
        
        html += `
                <div class="results-actions">
                    <a href="${window.location.href}" class="button button-primary">Refresh Page</a>
                </div>
            </div>
        `;
        
        $('#bulk-results').html(html).show();
    }
    
    function formatBytes(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
    }
});