<?php
/**
 * Plugin Name: TinySharp Image Optimizer Connector
 * Plugin URI: https://tinysharp.com
 * Description: Connect your WordPress site to TinySharp API for automatic image optimization with smart token management
 * Version: 1.0.1
 * Author: TinySharp
 * Author URI: https://tinysharp.com
 * License: GPL v2 or later
 */

if (!defined('ABSPATH')) {
    exit;
}

class TinySharpConnector {
    
    private $version = '1.0.1';
    private $backup_dir;
    private $stats_table;
    
    public function __construct() {
        global $wpdb;
        $this->backup_dir = WP_CONTENT_DIR . '/tinysharp-backups';
        $this->stats_table = $wpdb->prefix . 'tinysharp_stats';
        
        register_activation_hook(__FILE__, array($this, 'activate'));
        
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Auto-optimize on upload (if enabled)
        add_filter('wp_handle_upload', array($this, 'auto_optimize_upload'));
        
        // AJAX handlers
        add_action('wp_ajax_tinysharp_test_connection', array($this, 'ajax_test_connection'));
        add_action('wp_ajax_tinysharp_save_settings', array($this, 'ajax_save_settings'));
        add_action('wp_ajax_tinysharp_bulk_optimize', array($this, 'ajax_bulk_optimize'));
        add_action('wp_ajax_tinysharp_restore_image', array($this, 'ajax_restore_image'));
        
        // Media library columns
        add_filter('manage_media_columns', array($this, 'add_media_column'));
        add_action('manage_media_custom_column', array($this, 'show_media_column'), 10, 2);
    }
    
    public function activate() {
        global $wpdb;
        
        // Create backup directory
        if (!file_exists($this->backup_dir)) {
            wp_mkdir_p($this->backup_dir);
            file_put_contents($this->backup_dir . '/.htaccess', 'Deny from all');
        }
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Stats table only (settings use WordPress options)
        $sql = "CREATE TABLE IF NOT EXISTS {$this->stats_table} (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            attachment_id bigint(20) NOT NULL,
            original_size bigint(20) NOT NULL,
            optimized_size bigint(20) NOT NULL,
            saved_bytes bigint(20) NOT NULL,
            backup_path varchar(500),
            optimized_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY attachment_id (attachment_id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Default settings (only if not already set)
        if (get_option('tinysharp_auto_optimize') === false) {
            update_option('tinysharp_auto_optimize', '0');
        }
        if (get_option('tinysharp_create_backups') === false) {
            update_option('tinysharp_create_backups', '1');
        }
        if (get_option('tinysharp_max_width') === false) {
            update_option('tinysharp_max_width', '1920');
        }
        if (get_option('tinysharp_max_height') === false) {
            update_option('tinysharp_max_height', '1920');
        }
    }
    
    public function add_admin_menu() {
        add_menu_page(
            'TinySharp Optimizer',
            'TinySharp',
            'manage_options',
            'tinysharp',
            array($this, 'render_dashboard'),
            'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAiIGhlaWdodD0iMjAiIHZpZXdCb3g9IjAgMCAyMCAyMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cGF0aCBmaWxsPSIjYTdhYWFkIiBkPSJNMTAgMkM1LjYgMiAyIDUuNiAyIDEwczMuNiA4IDggOCA4LTMuNiA4LTgtMy42LTgtOC04em0wIDE0Yy0zLjMgMC02LTIuNy02LTZzMi43LTYgNi02IDYgMi43IDYgNi0yLjcgNi02IDZ6bTItOGgtNHY0aDR2LTR6Ii8+PC9zdmc+',
            65
        );
        
        add_submenu_page(
            'tinysharp',
            'Settings',
            'Settings',
            'manage_options',
            'tinysharp-settings',
            array($this, 'render_settings')
        );
        
        add_submenu_page(
            'tinysharp',
            'Bulk Optimizer',
            'Bulk Optimizer',
            'manage_options',
            'tinysharp-bulk',
            array($this, 'render_bulk_optimizer')
        );
    }
    
    public function enqueue_admin_scripts($hook) {
        if (strpos($hook, 'tinysharp') === false) {
            return;
        }
        
        wp_enqueue_style('tinysharp-admin', plugins_url('assets/admin-style.css', __FILE__), array(), $this->version);
        wp_enqueue_script('tinysharp-admin', plugins_url('assets/admin-script.js', __FILE__), array('jquery'), $this->version, true);
        
        wp_localize_script('tinysharp-admin', 'tinysharpData', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('tinysharp_nonce'),
            'api_url' => $this->get_option('api_url'),
            'api_key' => $this->get_option('api_key')
        ));
    }
    
    public function render_dashboard() {
        global $wpdb;
        
        $stats = $wpdb->get_row("
            SELECT 
                COUNT(*) as total_optimized,
                SUM(original_size) as total_original,
                SUM(optimized_size) as total_optimized_size,
                SUM(saved_bytes) as total_saved
            FROM {$this->stats_table}
        ");
        
        $api_key = $this->get_option('api_key');
        $tokens_data = $this->check_tokens();
        
        ?>
        <div class="wrap tinysharp-dashboard">
            <h1>TinySharp Image Optimizer</h1>
            
            <?php if (empty($api_key)): ?>
                <div class="tinysharp-welcome">
                    <div class="welcome-card">
                        <h2>Welcome to TinySharp!</h2>
                        <p>Connect your site to TinySharp API to start optimizing images automatically.</p>
                        <a href="<?php echo admin_url('admin.php?page=tinysharp-settings'); ?>" class="button button-primary button-hero">
                            Configure API Connection
                        </a>
                    </div>
                </div>
            <?php else: ?>
                
                <!-- Token Status -->
                <div class="tinysharp-token-status">
                    <?php if ($tokens_data && !is_wp_error($tokens_data)): ?>
                        <?php
                        $remaining = $tokens_data['remaining_tokens'];
                        $total = $tokens_data['total_tokens'];
                        $percentage = ($remaining / $total) * 100;
                        $status_class = $percentage > 50 ? 'good' : ($percentage > 20 ? 'warning' : 'critical');
                        ?>
                        <div class="token-card status-<?php echo $status_class; ?>">
                            <div class="token-icon">🎟️</div>
                            <div class="token-info">
                                <div class="token-plan"><?php echo strtoupper($tokens_data['plan']); ?> Plan</div>
                                <div class="token-count">
                                    <span class="remaining"><?php echo number_format($remaining); ?></span>
                                    <span class="separator">/</span>
                                    <span class="total"><?php echo number_format($total); ?></span>
                                </div>
                                <div class="token-label">Tokens Remaining</div>
                            </div>
                            <div class="token-progress">
                                <div class="progress-bar">
                                    <div class="progress-fill" style="width: <?php echo $percentage; ?>%"></div>
                                </div>
                            </div>
                        </div>
                        
                        <?php if ($remaining < 50): ?>
                            <div class="token-warning">
                                <strong>⚠️ Running Low on Tokens</strong>
                                <p>You have less than 50 tokens remaining. Consider upgrading your plan.</p>
                                <a href="<?php echo $tokens_data['upgrade_url'] ?? '#'; ?>" class="button" target="_blank">Upgrade Plan</a>
                            </div>
                        <?php endif; ?>
                    <?php else: ?>
                        <div class="token-error">
                            <strong>Connection Error</strong>
                            <p>Unable to fetch token data. Check your API settings.</p>
                        </div>
                    <?php endif; ?>
                </div>
                
                <!-- Stats Grid -->
                <div class="tinysharp-stats-grid">
                    <div class="stat-card">
                        <div class="stat-icon">🖼️</div>
                        <div class="stat-value"><?php echo number_format($stats->total_optimized ?? 0); ?></div>
                        <div class="stat-label">Images Optimized</div>
                    </div>
                    
                    <div class="stat-card">
                        <div class="stat-icon">💾</div>
                        <div class="stat-value"><?php echo $this->format_bytes($stats->total_saved ?? 0); ?></div>
                        <div class="stat-label">Total Saved</div>
                    </div>
                    
                    <div class="stat-card">
                        <div class="stat-icon">📊</div>
                        <div class="stat-value">
                            <?php 
                            $reduction = $stats->total_original > 0 
                                ? round((($stats->total_saved / $stats->total_original) * 100), 1)
                                : 0;
                            echo $reduction . '%';
                            ?>
                        </div>
                        <div class="stat-label">Average Reduction</div>
                    </div>
                    
                    <div class="stat-card">
                        <div class="stat-icon">📁</div>
                        <div class="stat-value"><?php echo $this->format_bytes($stats->total_original ?? 0); ?></div>
                        <div class="stat-label">Original Size</div>
                    </div>
                </div>
                
                <!-- Quick Actions -->
                <div class="tinysharp-quick-actions">
                    <h2>Quick Actions</h2>
                    <div class="action-buttons">
                        <a href="<?php echo admin_url('admin.php?page=tinysharp-bulk'); ?>" class="action-btn primary">
                            <span class="btn-icon">⚡</span>
                            <span class="btn-text">Bulk Optimize Images</span>
                        </a>
                        <a href="<?php echo admin_url('admin.php?page=tinysharp-settings'); ?>" class="action-btn">
                            <span class="btn-icon">⚙️</span>
                            <span class="btn-text">Configure Settings</span>
                        </a>
                        <a href="<?php echo admin_url('upload.php'); ?>" class="action-btn">
                            <span class="btn-icon">📂</span>
                            <span class="btn-text">Media Library</span>
                        </a>
                    </div>
                </div>
                
            <?php endif; ?>
        </div>
        <?php
    }
    
    public function render_settings() {
        // Get current settings
        $api_url = $this->get_option('api_url');
        $api_key = $this->get_option('api_key');
        $website = $this->get_option('website', home_url());
        $auto_optimize = $this->get_option('auto_optimize', '0');
        $create_backups = $this->get_option('create_backups', '1');
        $max_width = $this->get_option('max_width', '1920');
        $max_height = $this->get_option('max_height', '1920');
        
        ?>
        <div class="wrap tinysharp-settings">
            <h1>TinySharp Settings</h1>
            
            <div class="settings-container">
                <form id="tinysharp-settings-form">
                    
                    <!-- API Configuration -->
                    <div class="settings-section">
                        <h2>API Configuration</h2>
                        <p class="description">Connect your WordPress site to TinySharp API</p>
                        
                        <table class="form-table">
                            <tr>
                                <th><label for="api_url">API URL</label></th>
                                <td>
                                    <input type="url" name="api_url" id="api_url" value="<?php echo esc_attr($api_url); ?>" class="regular-text" placeholder="https://tinysharp.com/wp-json/sio/v1">
                                    <p class="description">Your TinySharp API endpoint URL</p>
                                </td>
                            </tr>
                            <tr>
                                <th><label for="api_key">API Key</label></th>
                                <td>
                                    <input type="text" name="api_key" id="api_key" value="<?php echo esc_attr($api_key); ?>" class="regular-text" placeholder="sio_...">
                                    <button type="button" id="test-connection" class="button">Test Connection</button>
                                    <p class="description">Get your API key from TinySharp dashboard</p>
                                    <div id="connection-result"></div>
                                </td>
                            </tr>
                            <tr>
                                <th><label for="website">Website URL</label></th>
                                <td>
                                    <input type="url" name="website" id="website" value="<?php echo esc_attr($website); ?>" class="regular-text">
                                    <p class="description">Your website URL (for tracking)</p>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                    <!-- Optimization Settings -->
                    <div class="settings-section">
                        <h2>Optimization Settings</h2>
                        
                        <table class="form-table">
                            <tr>
                                <th><label for="auto_optimize">Auto-Optimize Uploads</label></th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="auto_optimize" id="auto_optimize" value="1" <?php checked($auto_optimize, '1'); ?>>
                                        Automatically optimize images when uploaded to Media Library
                                    </label>
                                </td>
                            </tr>
                            <tr>
                                <th><label for="create_backups">Create Backups</label></th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="create_backups" id="create_backups" value="1" <?php checked($create_backups, '1'); ?>>
                                        Keep original images as backups (recommended)
                                    </label>
                                    <p class="description">Backups stored in: <code><?php echo $this->backup_dir; ?></code></p>
                                </td>
                            </tr>
                            <tr>
                                <th><label>Maximum Dimensions</label></th>
                                <td>
                                    <input type="number" name="max_width" id="max_width" value="<?php echo esc_attr($max_width); ?>" class="small-text"> x
                                    <input type="number" name="max_height" id="max_height" value="<?php echo esc_attr($max_height); ?>" class="small-text"> px
                                    <p class="description">Images larger than this will be resized (maintains aspect ratio)</p>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                    <p class="submit">
                        <button type="submit" class="button button-primary">Save Settings</button>
                    </p>
                </form>
            </div>
        </div>
        <?php
    }
    
    public function render_bulk_optimizer() {
    global $wpdb;
    
    // Check if stats table exists
    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$this->stats_table}'");
    
    if (!$table_exists) {
        echo '<div class="notice notice-error"><p><strong>Error:</strong> Stats table does not exist. Try deactivating and reactivating the plugin.</p></div>';
    }
    
    // PRVO proveri tokene
    $tokens_data = $this->check_tokens();
    $has_tokens = false;
    $tokens_remaining = 0;
    
    if ($tokens_data && !is_wp_error($tokens_data)) {
        $tokens_remaining = $tokens_data['remaining_tokens'];
        $has_tokens = $tokens_remaining > 0;
    }
    
    // Get ALL images
    $all_images = $wpdb->get_results("
        SELECT ID, post_title, guid
        FROM {$wpdb->posts}
        WHERE post_type = 'attachment'
        AND post_mime_type LIKE 'image/%'
        ORDER BY ID DESC
    ");
    
    // Get optimized image IDs
    $optimized_ids = $wpdb->get_col("SELECT attachment_id FROM {$this->stats_table}");
    
    // Filter out optimized images
    $attachments = array_filter($all_images, function($img) use ($optimized_ids) {
        return !in_array($img->ID, $optimized_ids);
    });
    
    $total_unoptimized = count($attachments);
    
    // Limit to first 100 for display
    $attachments = array_slice($attachments, 0, 100);
    
    ?>
    <div class="wrap tinysharp-bulk">
        <h1>Bulk Image Optimizer</h1>
        
        <div class="bulk-container">
            <?php if (!$has_tokens): ?>
                <!-- NO TOKENS WARNING -->
                <div class="bulk-no-tokens">
                    <div class="icon">⚠️</div>
                    <h2>No Tokens Available</h2>
                    <p>You don't have any optimization tokens. Please purchase a plan to start optimizing images.</p>
                    
                    <?php if ($tokens_data && !is_wp_error($tokens_data)): ?>
                        <div class="token-info">
                            <strong>Current Plan:</strong> <?php echo strtoupper($tokens_data['plan']); ?><br>
                            <strong>Tokens Used:</strong> <?php echo number_format($tokens_data['used_tokens']); ?> / <?php echo number_format($tokens_data['total_tokens']); ?>
                        </div>
                    <?php endif; ?>
                    
                    <div class="bulk-actions" style="margin-top: 30px;">
                        <?php
                        $api_url = $this->get_option('api_url', '');
                        $base_url = preg_replace('#/wp-json/sio/v1.*#', '', $api_url);
                        ?>
                        <a href="<?php echo $base_url; ?>/pricing-plans/" class="button button-primary button-hero" target="_blank">
                            Purchase Tokens
                        </a>
                        <a href="<?php echo admin_url('admin.php?page=tinysharp-settings'); ?>" class="button button-secondary">
                            Check API Settings
                        </a>
                    </div>
                </div>
                
            <?php elseif ($total_unoptimized > 0): ?>
                <div class="bulk-header">
                    <div class="bulk-stats">
                        <div class="stat">
                            <strong><?php echo number_format($total_unoptimized); ?></strong> images ready to optimize
                        </div>
                        <div class="stat tokens-info">
                            <strong><?php echo number_format($tokens_remaining); ?></strong> tokens available
                        </div>
                    </div>
                    <button id="start-bulk-optimize" class="button button-primary button-hero">
                        Start Bulk Optimization
                    </button>
                </div>
                
                <div id="bulk-progress" class="bulk-progress" style="display: none;">
                    <div class="progress-header">
                        <h3>Optimizing Images...</h3>
                        <div class="progress-stats">
                            <span id="progress-current">0</span> / <span id="progress-total"><?php echo $total_unoptimized; ?></span>
                        </div>
                    </div>
                    <div class="progress-bar-container">
                        <div id="progress-bar" class="progress-bar-fill"></div>
                    </div>
                    <div class="progress-info">
                        <div class="info-item">
                            <span class="label">Current:</span>
                            <span id="current-image">-</span>
                        </div>
                        <div class="info-item">
                            <span class="label">Saved:</span>
                            <span id="total-saved">0 KB</span>
                        </div>
                        <div class="info-item">
                            <span class="label">Tokens Used:</span>
                            <span id="tokens-used">0</span>
                        </div>
                        <div class="info-item">
                            <span class="label">Tokens Left:</span>
                            <span id="tokens-remaining"><?php echo number_format($tokens_remaining); ?></span>
                        </div>
                    </div>
                </div>
                
                <div id="bulk-results" class="bulk-results" style="display: none;"></div>
                
                <div class="images-list">
                    <h3>Images to Optimize (showing first 20)</h3>
                    <div class="images-grid">
                        <?php foreach (array_slice($attachments, 0, 20) as $attachment): ?>
                            <div class="image-item" data-id="<?php echo $attachment->ID; ?>">
                                <img src="<?php echo wp_get_attachment_thumb_url($attachment->ID); ?>" alt="">
                                <div class="image-title"><?php echo esc_html($attachment->post_title ?: 'Untitled'); ?></div>
                            </div>
                        <?php endforeach; ?>
                        <?php if ($total_unoptimized > 20): ?>
                            <div class="image-item more">
                                <div class="more-count">+<?php echo ($total_unoptimized - 20); ?> more</div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                
            <?php else: ?>
                <div class="no-images">
                    <div class="icon">✅</div>
                    <h2>All Images Optimized!</h2>
                    <p>All images in your media library have been optimized.</p>
                    <a href="<?php echo admin_url('upload.php'); ?>" class="button button-primary">View Media Library</a>
                </div>
            <?php endif; ?>
        </div>
        
        <script>
        var bulkImageIds = <?php echo json_encode(wp_list_pluck($attachments, 'ID')); ?>;
        var initialTokensRemaining = <?php echo $tokens_remaining; ?>;
        </script>
    </div>
    
    <style>
    .bulk-no-tokens {
        background: white;
        border: 2px solid #dc3545;
        border-radius: 12px;
        padding: 60px 40px;
        text-align: center;
        max-width: 600px;
        margin: 40px auto;
    }
    .bulk-no-tokens .icon {
        font-size: 5em;
        margin-bottom: 20px;
    }
    .bulk-no-tokens h2 {
        color: #dc3545;
        font-size: 2em;
        margin-bottom: 15px;
    }
    .bulk-no-tokens p {
        font-size: 1.2em;
        color: #666;
        margin-bottom: 25px;
    }
    .bulk-no-tokens .token-info {
        background: #f8f9fa;
        padding: 20px;
        border-radius: 8px;
        margin: 20px 0;
        text-align: left;
        font-size: 1em;
        line-height: 1.8;
    }
    .bulk-stats {
        display: flex;
        gap: 30px;
        align-items: center;
    }
    .bulk-stats .tokens-info {
        color: #0969da;
    }
    </style>
    <?php
}
    
    // AJAX Handlers
    
    public function ajax_test_connection() {
        check_ajax_referer('tinysharp_nonce', 'nonce');
        
        $api_url = sanitize_text_field($_POST['api_url']);
        $api_key = sanitize_text_field($_POST['api_key']);
        
        $response = wp_remote_get($api_url . '/tokens', array(
            'headers' => array('X-API-Key' => $api_key)
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }
        
        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($code === 200) {
            wp_send_json_success(array(
                'message' => 'Connection successful!',
                'data' => $body
            ));
        } else {
            wp_send_json_error(array(
                'message' => 'Connection failed: ' . ($body['message'] ?? 'Unknown error')
            ));
        }
    }
    
    public function ajax_save_settings() {
        check_ajax_referer('tinysharp_nonce', 'nonce');
        
        // Save settings using WordPress options
        $this->set_option('api_url', sanitize_text_field($_POST['api_url']));
        $this->set_option('api_key', sanitize_text_field($_POST['api_key']));
        $this->set_option('website', esc_url_raw($_POST['website']));
        
        // Handle checkboxes properly
        $auto_optimize = isset($_POST['auto_optimize']) && $_POST['auto_optimize'] === '1' ? '1' : '0';
        $create_backups = isset($_POST['create_backups']) && $_POST['create_backups'] === '1' ? '1' : '0';
        
        $this->set_option('auto_optimize', $auto_optimize);
        $this->set_option('create_backups', $create_backups);
        $this->set_option('max_width', intval($_POST['max_width']));
        $this->set_option('max_height', intval($_POST['max_height']));
        
        wp_send_json_success(array('message' => 'Settings saved successfully!'));
    }
    
    public function ajax_bulk_optimize() {
        check_ajax_referer('tinysharp_nonce', 'nonce');
        
        $attachment_id = intval($_POST['attachment_id']);
        $result = $this->optimize_image($attachment_id);
        
        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }
        
        wp_send_json_success($result);
    }
    
    public function ajax_restore_image() {
        check_ajax_referer('tinysharp_nonce', 'nonce');
        
        $attachment_id = intval($_POST['attachment_id']);
        $result = $this->restore_original($attachment_id);
        
        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }
        
        wp_send_json_success(array('message' => 'Image restored successfully'));
    }
    
    // Core Functions
    
    public function optimize_image($attachment_id) {
        $file_path = get_attached_file($attachment_id);
        
        if (!file_exists($file_path)) {
            return new WP_Error('file_not_found', 'Image file not found');
        }
        
        $original_size = filesize($file_path);
        
        // Create backup if enabled
        $backup_path = null;
        if ($this->get_option('create_backups') === '1') {
            $backup_path = $this->create_backup($file_path, $attachment_id);
        }
        
        // Prepare API request
        $api_url = $this->get_option('api_url');
        $api_key = $this->get_option('api_key');
        
        if (empty($api_url) || empty($api_key)) {
            return new WP_Error('missing_config', 'API URL or API Key not configured');
        }
        
        // WordPress way of sending files
        $boundary = wp_generate_password(24);
        $body = '';
        
        // Add file to multipart body
        $body .= "--{$boundary}\r\n";
        $body .= 'Content-Disposition: form-data; name="image"; filename="' . basename($file_path) . '"' . "\r\n";
        $body .= 'Content-Type: ' . mime_content_type($file_path) . "\r\n\r\n";
        $body .= file_get_contents($file_path) . "\r\n";
        $body .= "--{$boundary}--\r\n";
        
        $response = wp_remote_post($api_url . '/optimize', array(
            'headers' => array(
                'X-API-Key' => $api_key,
                'Content-Type' => 'multipart/form-data; boundary=' . $boundary
            ),
            'body' => $body,
            'timeout' => 60
        ));
        
        if (is_wp_error($response)) {
            error_log('TinySharp API Error: ' . $response->get_error_message());
            return $response;
        }
        
        $http_code = wp_remote_retrieve_response_code($response);
        $body_response = wp_remote_retrieve_body($response);
        
        if ($http_code !== 200) {
            return new WP_Error('api_error', 'API returned error code: ' . $http_code);
        }
        
        $data = json_decode($body_response, true);
        
        if (!isset($data['success']) || !$data['success']) {
            return new WP_Error('optimization_failed', $data['message'] ?? 'Optimization failed');
        }
        
        // Save optimized image
        $optimized_content = base64_decode($data['image']);
        file_put_contents($file_path, $optimized_content);
        
        // Regenerate thumbnails
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        $attach_data = wp_generate_attachment_metadata($attachment_id, $file_path);
        wp_update_attachment_metadata($attachment_id, $attach_data);
        
        $optimized_size = filesize($file_path);
        $saved = $original_size - $optimized_size;
        
        // Save stats
        global $wpdb;
        $wpdb->replace($this->stats_table, array(
            'attachment_id' => $attachment_id,
            'original_size' => $original_size,
            'optimized_size' => $optimized_size,
            'saved_bytes' => $saved,
            'backup_path' => $backup_path
        ));
        
        return array(
            'attachment_id' => $attachment_id,
            'original_size' => $original_size,
            'optimized_size' => $optimized_size,
            'saved' => $saved,
            'tokens_remaining' => $data['tokens_remaining'] ?? 0
        );
    }
    
    public function auto_optimize_upload($upload) {
        if ($this->get_option('auto_optimize') !== '1') {
            return $upload;
        }
        
        if (!isset($upload['file']) || strpos($upload['type'], 'image/') !== 0) {
            return $upload;
        }
        
        $attachment_id = attachment_url_to_postid($upload['url']);
        if ($attachment_id) {
            $this->optimize_image($attachment_id);
        }
        
        return $upload;
    }
    
    private function create_backup($file_path, $attachment_id) {
        $backup_filename = $attachment_id . '_' . basename($file_path);
        $backup_path = $this->backup_dir . '/' . $backup_filename;
        
        copy($file_path, $backup_path);
        
        return $backup_path;
    }
    
    private function restore_original($attachment_id) {
        global $wpdb;
        
        $stat = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->stats_table} WHERE attachment_id = %d",
            $attachment_id
        ));
        
        if (!$stat || !$stat->backup_path || !file_exists($stat->backup_path)) {
            return new WP_Error('backup_not_found', 'Backup not found');
        }
        
        $file_path = get_attached_file($attachment_id);
        copy($stat->backup_path, $file_path);
        
        $wpdb->delete($this->stats_table, array('attachment_id' => $attachment_id));
        
        return true;
    }
    
    private function check_tokens() {
        $api_url = $this->get_option('api_url');
        $api_key = $this->get_option('api_key');
        
        if (empty($api_url) || empty($api_key)) {
            return null;
        }
        
        $response = wp_remote_get($api_url . '/tokens', array(
            'headers' => array('X-API-Key' => $api_key),
            'timeout' => 15
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        return json_decode(wp_remote_retrieve_body($response), true);
    }
    
    public function add_media_column($columns) {
        $columns['tinysharp'] = 'TinySharp';
        return $columns;
    }
    
    public function show_media_column($column_name, $attachment_id) {
        if ($column_name !== 'tinysharp') {
            return;
        }
        
        global $wpdb;
        $stat = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->stats_table} WHERE attachment_id = %d",
            $attachment_id
        ));
        
        if ($stat) {
            $saved_percent = round(($stat->saved_bytes / $stat->original_size) * 100);
            echo '<span style="color: #238636; font-weight: 600;">✓ Optimized</span><br>';
            echo '<small>Saved ' . $this->format_bytes($stat->saved_bytes) . ' (' . $saved_percent . '%)</small>';
        } else {
            echo '<span style="color: #656d76;">Not optimized</span>';
        }
    }
    
    // Helper Methods
    
    /**
     * Save option using WordPress options table
     */
    private function set_option($name, $value) {
        update_option('tinysharp_' . $name, $value);
    }
    
    /**
     * Get option from WordPress options table
     */
    private function get_option($name, $default = '') {
        return get_option('tinysharp_' . $name, $default);
    }
    
    private function format_bytes($bytes) {
        if ($bytes < 1024) return $bytes . ' B';
        if ($bytes < 1048576) return round($bytes / 1024, 2) . ' KB';
        return round($bytes / 1048576, 2) . ' MB';
    }
}

new TinySharpConnector();